
/*

All the methods in this class other than MyMath() have been
taken from the Cephes Math Library and Modified from C code to
Java code by Manfred Georg.

*/



public final class MyMath extends Object {

  private MyMath() {} // never instatiate

  private static final double
    MACHEP =  1.11022302462515654042E-16d,
    UFLOWTHRESH =  2.22507385850720138309E-308d,
    MAXLOG =  7.09782712893383996732E2d,
    MINLOG = -7.451332191019412076235E2d,
    MAXNUM =  1.79769313486231570815E308d,   /* 2**1024*(1-MACHEP) */
    PI     =  3.14159265358979323846d,       /* pi */
    PIO2   =  1.57079632679489661923d,       /* pi/2 */
    PIO4   =  7.85398163397448309616E-1d,    /* pi/4 */
    SQRT2  =  1.41421356237309504880d,       /* sqrt(2) */
    SQRTH  =  7.07106781186547524401E-1d,    /* sqrt(2)/2 */
    LOG2E  =  1.4426950408889634073599d,     /* 1/log(2) */
    SQ2OPI =  7.9788456080286535587989E-1d,  /* sqrt( 2/pi ) */
    LOGE2  =  6.93147180559945309417E-1d,    /* log(2) */
    LOGSQ2 =  3.46573590279972654709E-1d,    /* log(2)/2 */
    THPIO4 =  2.35619449019234492885d,       /* 3*pi/4 */
    TWOOPI =  6.36619772367581343075535E-1d, /* 2/pi */
    s2pi   =  2.50662827463100050242E0d,          /* sqrt(2pi) */
    SQTPI  =  2.50662827463100050242E0d,          /* sqrt(2pi) */
    INFINITY = Double.POSITIVE_INFINITY,
    NAN = Double.NaN,
    NEGZERO = -0d,
    MAXGAM = 171.624376956302725d,
    big = 4.503599627370496e15d,
    biginv =  2.22044604925031308085e-16d,
    LOGPI = 1.14472988584940017414;

/*                                                      polevl.c
 *                                                      p1evl.c
 *
 *      Evaluate polynomial
 *
 *
 *
 * SYNOPSIS:
 *
 * int N;
 * double x, y, coef[N+1], polevl[];
 *
 * y = polevl( x, coef, N );
 *
 *
 *
 * DESCRIPTION:
 *
 * Evaluates polynomial of degree N:
 *
 *                     2          N
 * y  =  C  + C x + C x  +...+ C x
 *        0    1     2          N
 *
 * Coefficients are stored in reverse order:
 *
 * coef[0] = C  , ..., coef[N] = C  .
 *            N                   0
 *
 *  The function p1evl() assumes that coef[N] = 1.0 and is
 * omitted from the array.  Its calling arguments are
 * otherwise the same as polevl().
 */

  public static double polevl( double x, double[] coef) {
    int i=coef.length-1, p=0;
    double ans=coef[0];
    /*if(i!=N)
      System.out.println("polevl("+x+", "+coef+", "+N+");  i= "+i+"\tN= "+N);
    else
      System.out.println("polevl();");*/
    do
      ans = ans * x  +  coef[++p];
    while( --i!=0 );

    return ans;
  }

  /*                                          N
   * Evaluate polynomial when coefficient of x  is 1.0.
   * Otherwise same as polevl.
   */

  public static double p1evl( double x, double[] coef) {
    int i=coef.length, p=0;
    double ans=x+coef[0];
    /*if(i!=N)
      System.out.println("p1evl("+x+", "+coef+", "+N+");  i= "+i+"\tN= "+N);
    else
      System.out.println("p1evl())");*/

    do
      ans = ans * x  +  coef[++p];
    while( --i!=1 );  // leave early because of 1 at begining

    return ans;
  }

/*                                                      ndtr.c
 *
 *      Normal distribution function
 *
 *
 *
 * SYNOPSIS:
 *
 * double x, y, ndtr();
 *
 * y = ndtr( x );
 *
 *
 *
 * DESCRIPTION:
 *
 * Returns the area under the Gaussian probability density
 * function, integrated from minus infinity to x:
 *
 *                            x
 *                             -
 *                   1        | |          2
 *    ndtr(x)  = ---------    |    exp( - t /2 ) dt
 *               sqrt(2pi)  | |
 *                           -
 *                          -inf.
 *
 *             =  ( 1 + erf(z) ) / 2
 *             =  erfc(z) / 2
 *
 * where z = x/sqrt(2). Computation is via the functions
 * erf and erfc.
 *
 *
 * ACCURACY:
 *
 *                      Relative error:
 * arithmetic   domain     # trials      peak         rms
 *    DEC      -13,0         8000       2.1e-15     4.8e-16
 *    IEEE     -13,0        30000       3.4e-14     6.7e-15
 *
 *
 * ERROR MESSAGES:
 *
 *   message         condition         value returned
 * erfc underflow    x > 37.519379347       0.0
 *
 */

  public static double ndtr(double a) {
    double x, y, z;
  
    x = a * SQRTH;
    z = Math.abs(x);
  
    if( z < SQRTH )
      y = 0.5 + 0.5 * erf(x);
    else {
      y = 0.5 * erfc(z);
      if( x > 0 )
        y = 1.0 - y;
    }
    return y;
  }

/*                                                      erf.c
 *
 *      Error function
 *
 *
 *
 * SYNOPSIS:
 *
 * double x, y, erf();
 *
 * y = erf( x );
 *
 *
 *
 * DESCRIPTION:
 *
 * The integral is
 *
 *                           x
 *                            -
 *                 2         | |          2
 *   erf(x)  =  --------     |    exp( - t  ) dt.
 *              sqrt(pi)   | |
 *                          -
 *                           0
 *
 * The magnitude of x is limited to 9.231948545 for DEC
 * arithmetic; 1 or -1 is returned outside this range.
 *
 * For 0 <= |x| < 1, erf(x) = x * P4(x**2)/Q5(x**2); otherwise
 * erf(x) = 1 - erfc(x).
 *
 *
 *
 * ACCURACY:
 *
 *                      Relative error:
 * arithmetic   domain     # trials      peak         rms
 *    DEC       0,1         14000       4.7e-17     1.5e-17
 *    IEEE      0,1         30000       3.7e-16     1.0e-16
 *
 */


  public static double erf(double x) {
    //System.out.println("erf();");
    final double T[] = {
      9.60497373987051638749E0d,
      9.00260197203842689217E1d,
      2.23200534594684319226E3d,
      7.00332514112805075473E3d,
      5.55923013010394962768E4d
    };

    final double U[] = {
      /* 1.00000000000000000000E0d,*/
      3.35617141647503099647E1d,
      5.21357949780152679795E2d,
      4.59432382970980127987E3d,
      2.26290000613890934246E4d,
      4.92673942608635921086E4d
    };

    double y, z;

    if( Math.abs(x) > 1d )
      return( 1d - erfc(x) );
    z = x * x;
    y = x * polevl( z, T ) / p1evl( z, U );
    return y;
  }

/*                                                      erfc.c
 *
 *      Complementary error function
 *
 *
 *
 * SYNOPSIS:
 *
 * double x, y, erfc();
 *
 * y = erfc( x );
 *
 *
 *
 * DESCRIPTION:


 *
 *
 *  1 - erf(x) =
 *
 *                           inf.
 *                             -
 *                  2         | |          2
 *   erfc(x)  =  --------     |    exp( - t  ) dt
 *               sqrt(pi)   | |
 *                           -
 *                            x
 *
 *
 * For small x, erfc(x) = 1 - erf(x); otherwise rational
 * approximations are computed.
 *
 *
 *
 * ACCURACY:
 *
 *                      Relative error:
 * arithmetic   domain     # trials      peak         rms
 *    DEC       0, 9.2319   12000       5.1e-16     1.2e-16
 *    IEEE      0,26.6417   30000       5.7e-14     1.5e-14
 *
 *
 * ERROR MESSAGES:
 *
 *   message         condition              value returned
 * erfc underflow    x > 9.231948545 (DEC)       0.0
 *
 *
 */
  public static double erfc(double a) {
    //System.out.println("erfc();");
    final double P[] = {
      2.46196981473530512524E-10d,
      5.64189564831068821977E-1d,
      7.46321056442269912687E0d,
      4.86371970985681366614E1d,
      1.96520832956077098242E2d,
      5.26445194995477358631E2d,
      9.34528527171957607540E2d,
      1.02755188689515710272E3d,
      5.57535335369399327526E2d
    };
    final double Q[] = {
      /* 1.00000000000000000000E0d,*/
      1.32281951154744992508E1d,
      8.67072140885989742329E1d,
      3.54937778887819891062E2d,
      9.75708501743205489753E2d,
      1.82390916687909736289E3d,
      2.24633760818710981792E3d,
      1.65666309194161350182E3d,
      5.57535340817727675546E2d
    };
    final double R[] = {
      5.64189583547755073984E-1d,
      1.27536670759978104416E0d,
      5.01905042251180477414E0d,
      6.16021097993053585195E0d,
      7.40974269950448939160E0d,
      2.97886665372100240670E0d
    };
    final double S[] = {
      /* 1.00000000000000000000E0d,*/
      2.26052863220117276590E0d,
      9.39603524938001434673E0d,
      1.20489539808096656605E1d,
      1.70814450747565897222E1d,
      9.60896809063285878198E0d,
      3.36907645100081516050E0d
    };

    double p,q,x,y,z;
    if( a < 0d )
      x = -a;
    else
      x = a;

    if( x < 1d )
      return( 1d - erf(a) );

    z = Math.exp(-a * a);

    if( x < 8d ) {
      p = polevl( x, P );
      q = p1evl( x, Q );
    }
    else {
      p = polevl( x, R );
      q = p1evl( x, S );
    }
    y = (z * p)/q;

    if( a < 0d )
      y = 2d - y;

    return y;
  }

/*                                                      ndtri.c
 *
 *      Inverse of Normal distribution function
 *
 *
 *
 * SYNOPSIS:
 *
 * double x, y, ndtri();
 *
 * x = ndtri( y );
 *
 *
 *
 * DESCRIPTION:
 *
 * Returns the argument, x, for which the area under the
 * Gaussian probability density function (integrated from
 * minus infinity to x) is equal to y.
 *
 *
 * For small arguments 0 < y < exp(-2), the program computes
 * z = sqrt( -2.0 * log(y) );  then the approximation is
 * x = z - log(z)/z  - (1/z) P(1/z) / Q(1/z).
 * There are two rational functions P/Q, one for 0 < y < exp(-32)
 * and the other for y up to exp(-2).  For larger arguments,
 * w = y - 0.5, and  x/sqrt(2pi) = w + w**3 R(w**2)/S(w**2)).
 *
 *
 * ACCURACY:
 *
 *                      Relative error:
 * arithmetic   domain        # trials      peak         rms
 *    DEC      0.125, 1         5500       9.5e-17     2.1e-17
 *    DEC      6e-39, 0.135     3500       5.7e-17     1.3e-17
 *    IEEE     0.125, 1        20000       7.2e-16     1.3e-16
 *    IEEE     3e-308, 0.135   50000       4.6e-16     9.8e-17
 *
 *
 * ERROR MESSAGES:
 *
 *   message         condition    value returned
 * ndtri domain       x <= 0        -MAXNUM
 * ndtri domain       x >= 1         MAXNUM
 *
 */

  public static double ndtri(double y0) {
    /* approximation for 0 <= |y - 0.5| <= 3/8 */
    final double P0[] = {
      -5.99633501014107895267E1d,
       9.80010754185999661536E1d,
      -5.66762857469070293439E1d,
       1.39312609387279679503E1d,
      -1.23916583867381258016E0d,
    };
    final double Q0[] = {
      /* 1.00000000000000000000E0d,*/
       1.95448858338141759834E0d,
       4.67627912898881538453E0d,
       8.63602421390890590575E1d,
      -2.25462687854119370527E2d,
       2.00260212380060660359E2d,
      -8.20372256168333339912E1d,
       1.59056225126211695515E1d,
      -1.18331621121330003142E0d,
    };
    /* Approximation for interval z = sqrt(-2 log y ) between 2 and 8
     * i.e., y between exp(-2) = .135 and exp(-32) = 1.27e-14.
     */
    final double P1[] = {
       4.05544892305962419923E0d,
       3.15251094599893866154E1d,
       5.71628192246421288162E1d,
       4.40805073893200834700E1d,
       1.46849561928858024014E1d,
       2.18663306850790267539E0d,
      -1.40256079171354495875E-1d,
      -3.50424626827848203418E-2d,
      -8.57456785154685413611E-4d,
    };
    final double Q1[] = {
      /*  1.00000000000000000000E0d,*/
       1.57799883256466749731E1d,
       4.53907635128879210584E1d,
       4.13172038254672030440E1d,
       1.50425385692907503408E1d,
       2.50464946208309415979E0d,
      -1.42182922854787788574E-1d,
      -3.80806407691578277194E-2d,
      -9.33259480895457427372E-4d,
    };
    /* Approximation for interval z = sqrt(-2 log y ) between 8 and 64
     * i.e., y between exp(-32) = 1.27e-14 and exp(-2048) = 3.67e-890.
     */

    final double P2[] = {
      3.23774891776946035970E0d,
      6.91522889068984211695E0d,
      3.93881025292474443415E0d,
      1.33303460815807542389E0d,
      2.01485389549179081538E-1d,
      1.23716634817820021358E-2d,
      3.01581553508235416007E-4d,
      2.65806974686737550832E-6d,
      6.23974539184983293730E-9d,
    };
    final double Q2[] = {
      /*  1.00000000000000000000E0d,*/
      6.02427039364742014255E0d,
      3.67983563856160859403E0d,
      1.37702099489081330271E0d,
      2.16236993594496635890E-1d,
      1.34204006088543189037E-2d,
      3.28014464682127739104E-4d,
      2.89247864745380683936E-6d,
      6.79019408009981274425E-9d,
    };
    double x, y, z, y2, x0, x1;
    boolean code=true;

    /*if( y0 <= 0.0 ) {
      mtherr( "ndtri", DOMAIN );
      return( -MAXNUM );
    }
    if( y0 >= 1.0 ) {
      mtherr( "ndtri", DOMAIN );
      return( MAXNUM );
    }*/
    y = y0;
    if( y > (1.0 - 0.13533528323661269189) ) { /* 0.135... = exp(-2) */
      y = 1.0 - y;
      code = false;
    }

    if( y > 0.13533528323661269189 ) {
      y = y - 0.5;
      y2 = y * y;
      x = y + y * (y2 * polevl( y2, P0)/p1evl( y2, Q0 ));
      x = x * s2pi;
      return(x);
    }

    x = Math.sqrt( -2.0 * Math.log(y) );
    x0 = x - Math.log(x)/x;

    z = 1.0/x;
    if( x < 8.0 ) /* y > exp(-32) = 1.2664165549e-14 */
      x1 = z * polevl( z, P1 )/p1evl( z, Q1 );
    else
      x1 = z * polevl( z, P2 )/p1evl( z, Q2 );
    x = x0 - x1;
    if( code )
      x = -x;
    return x;
  }

/*                                                      stdtr.c
 *
 *      Student's t distribution
 *
 *
 *
 * SYNOPSIS:
 *
 * double t, stdtr();
 * short k;
 *
 * y = stdtr( k, t );
 *
 *
 * DESCRIPTION:
 *
 * Computes the integral from minus infinity to t of the Student
 * t distribution with integer k > 0 degrees of freedom:
 *
 *                                      t
 *                                      -
 *                                     | |
 *              -                      |         2   -(k+1)/2
 *             | ( (k+1)/2 )           |  (     x   )
 *       ----------------------        |  ( 1 + --- )        dx
 *                     -               |  (      k  )
 *       sqrt( k pi ) | ( k/2 )        |
 *                                   | |
 *                                    -
 *                                   -inf.
 *
 * Relation to incomplete beta integral:
 *
 *        1 - stdtr(k,t) = 0.5 * incbet( k/2, 1/2, z )
 * where
 *        z = k/(k + t**2).
 *
 * For t < -2, this is the method of computation.  For higher t,
 * a direct method is derived from integration by parts.
 * Since the function is symmetric about t=0, the area under the
 * right tail of the density is found by calling the function
 * with -t instead of t.
 *
 * ACCURACY:
 *
 * Tested at random 1 <= k <= 25.  The "domain" refers to t.
 *                      Relative error:
 * arithmetic   domain     # trials      peak         rms
 *    IEEE     -100,-2      50000       5.9e-15     1.4e-15
 *    IEEE     -2,100      500000       2.7e-15     4.9e-17
 */

  public static double stdtr( double t, int k )
         throws MyArithmeticException {
    double x, rk, z, f, tz, p, xsqk;
    int j;

    if( k <= 0 ) {
      throw new MyArithmeticException();
    }

    if( t == 0d )
      return 0.5d;

    if( t < -2.0d ) {
      rk = k;
      z = rk / (rk + t * t);
      p = 0.5d * incbet( 0.5d*rk, 0.5d, z );
      return p;
    }

/*      compute integral from -t to + t */

    if( t < 0d )
      x = -t;
    else
      x = t;

    rk = k; /* degrees of freedom */
    z = 1.0d + ( x * x )/rk;

    /* test if k is odd or even */
    if( (k & 1) != 0) {
      /*      computation for odd k   */

      xsqk = x/Math.sqrt(rk);
      p = Math.atan( xsqk );
      if( k > 1 ) {
        f = 1.0d;
        tz = 1.0d;
        j = 3;
        while(  (j<=(k-2)) && ( (tz/f) > MACHEP )  ) {
          tz *= (j-1)/( z * j );
          f += tz;
          j += 2;
        }
        p += f * xsqk/z;
      }
      p *= TWOOPI;
    }
    else {
      /*      computation for even k  */

      f = 1.0;
      tz = 1.0;
      j = 2;

      while(  ( j <= (k-2) ) && ( (tz/f) > MACHEP )  ) {
        tz *= (j - 1)/( z * j );
        f += tz;
        j += 2;
      }
      p = f * x/Math.sqrt(z*rk);
    }

    /*      common exit     */


    if( t < 0 )
      p = -p; /* note destruction of relative accuracy */

    p = 0.5 + 0.5 * p;
    return p;
  }

/*                                                      stdtri.c
 *
 *      Functional inverse of Student's t distribution
 *
 *
 *
 * SYNOPSIS:
 *
 * double p, t, stdtri();
 * int k;
 *
 * t = stdtri( p, k );
 *
 *
 * DESCRIPTION:
 *
 * Given probability p, finds the argument t such that stdtr(k,t)
 * is equal to p.
 *
 * ACCURACY:
 *
 * Tested at random 1 <= k <= 100.  The "domain" refers to p:
 *                      Relative error:
 * arithmetic   domain     # trials      peak         rms
 *    IEEE    .001,.999     25000       5.7e-15     8.0e-16
 *    IEEE    10^-6,.001    25000       2.0e-12     2.9e-14
 */

  public static double stdtri( double p, int k )
         throws MyArithmeticException {
    double t, rk, z;
    int rflg;

    if( k <= 0 || p <= 0.0d || p >= 1.0d ) {
      throw new MyArithmeticException();
    }

    rk = k;

    if( p > 0.25d && p < 0.75d ) {
      if( p == 0.5d )
        return 0d ;
      z = 1.0d - 2.0d * p;
      z = incbi( 0.5d, 0.5d*rk, Math.abs(z) );
      t = Math.sqrt( rk*z/(1.0d-z) );
      if( p < 0.5 )
        t = -t;
      return t;
    }
    rflg = -1;
    if( p >= 0.5d) {
      p = 1.0d - p;
      rflg = 1;
    }
    z = incbi( 0.5*rk, 0.5, 2.0*p );

    if( MAXNUM * z < rk )
      return rflg * MAXNUM;
    t = Math.sqrt( rk/z - rk );
    return rflg * t;
  }

/*                                                      incbet.c
 *
 *      Incomplete beta integral
 *
 *
 * SYNOPSIS:
 *
 * double a, b, x, y, incbet();
 *
 * y = incbet( a, b, x );
 *
 *
 * DESCRIPTION:
 *
 * Returns incomplete beta integral of the arguments, evaluated
 * from zero to x.  The function is defined as
 *
 *                  x
 *     -            -
 *    | (a+b)      | |  a-1     b-1
 *  -----------    |   t   (1-t)   dt.
 *   -     -     | |
 *  | (a) | (b)   -
 *                 0
 *
 * The domain of definition is 0 <= x <= 1.  In this
 * implementation a and b are restricted to positive values.
 * The integral from x to 1 may be obtained by the symmetry
 * relation
 *
 *    1 - incbet( a, b, x )  =  incbet( b, a, 1-x ).
 *
 * The integral is evaluated by a continued fraction expansion
 * or, when b*x is small, by a power series.
 *
 * ACCURACY:
 *
 * Tested at uniformly distributed random points (a,b,x) with a and b
 * in "domain" and x between 0 and 1.
 *                                        Relative error
 * arithmetic   domain     # trials      peak         rms
 *    IEEE      0,5         10000       6.9e-15     4.5e-16
 *    IEEE      0,85       250000       2.2e-13     1.7e-14
 *    IEEE      0,1000      30000       5.3e-12     6.3e-13
 *    IEEE      0,10000    250000       9.3e-11     7.1e-12
 *    IEEE      0,100000    10000       8.7e-10     4.8e-11
 * Outputs smaller than the IEEE gradual underflow threshold
 * were excluded from these statistics.
 *
 * ERROR MESSAGES:
 *   message         condition      value returned
 * incbet domain      x<0, x>1          0.0
 * incbet underflow                     0.0
 */

  public static double incbet( double aa, double bb, double xx )
         throws MyArithmeticException {
    double a, b, t, x, xc, w, y;
    int flag=0;

    if( aa <= 0.0d || bb <= 0.0d )
      throw new MyArithmeticException();

    if( (xx <= 0.0d) || ( xx >= 1.0d) ) {
      if( xx == 0.0d )
        return 0d;
      if( xx == 1.0d )
        return( 1.0d );
      throw new MyArithmeticException();
    }
    if( (bb * xx) <= 1.0d && xx <= 0.95d) {
      t = pseries(aa, bb, xx);
        return incbetDone(flag,t);
    }

    w = 1.0d - xx;

    /* Reverse a and b if x is greater than the mean. */
    if( xx > (aa/(aa+bb)) ) {
      flag = 1;
      a = bb;
      b = aa;
      xc = xx;
      x = w;
    }
    else {
      a = aa;
      b = bb;
      xc = w;
      x = xx;
    }

    if( flag == 1 && (b * x) <= 1.0d && x <= 0.95d) {
      t = pseries(a, b, x);
      return incbetDone(flag, t);
    }

    /* Choose expansion for better convergence. */
    y = x * (a+b-2.0d) - (a-1.0d);
    if( y < 0.0d )
      w = incbcf( a, b, x );
    else
      w = incbd( a, b, x ) / xc;

    /* Multiply w by the factor
     a      b   _             _     _
    x  (1-x)   | (a+b) / ( a | (a) | (b) ) .   */

    y = a * Math.log(x);
    t = b * Math.log(xc);
    if( (a+b) < MAXGAM && Math.abs(y) < MAXLOG && Math.abs(t) < MAXLOG ) {
      t = Math.pow(xc,b);
      t *= Math.pow(x,a);
      t /= a;
      t *= w;
      t *= gamma(a+b) / (gamma(a) * gamma(b));
      return incbetDone(flag,t);
    }
    /* Resort to logarithms.  */
    y += t + lgam(a+b) - lgam(a) - lgam(b);
    y += Math.log(w/a);
    if( y < MINLOG )
      t = 0d;
    else
      t = Math.exp(y);

    return incbetDone(flag,t);

  }

  private static double incbetDone(int flag, double t) {
    if( flag == 1 ) {
      if( t <= MACHEP )
        t = 1.0d - MACHEP;
      else
        t = 1.0d - t;
    }
    return t;
  }

  /* Continued fraction expansion #1
   * for incomplete beta integral
   */

  private static double incbcf( double a, double b, double x ) {
    double xk, pk, pkm1, pkm2, qk, qkm1, qkm2;
    double k1, k2, k3, k4, k5, k6, k7, k8;
    double r, t, ans, thresh;
    int n;

    k1 = a;
    k2 = a + b;
    k3 = a;
    k4 = a + 1.0;
    k5 = 1.0;
    k6 = b - 1.0;
    k7 = k4;
    k8 = a + 2.0;

    pkm2 = 0.0;
    qkm2 = 1.0;
    pkm1 = 1.0;
    qkm1 = 1.0;
    ans = 1.0;
    r = 1.0;
    n = 0;
    thresh = 3.0 * MACHEP;
    do {

      xk = -( x * k1 * k2 )/( k3 * k4 );
      pk = pkm1 +  pkm2 * xk;
      qk = qkm1 +  qkm2 * xk;
      pkm2 = pkm1;
      pkm1 = pk;
      qkm2 = qkm1;
      qkm1 = qk;

      xk = ( x * k5 * k6 )/( k7 * k8 );
      pk = pkm1 +  pkm2 * xk;
      qk = qkm1 +  qkm2 * xk;
      pkm2 = pkm1;
      pkm1 = pk;
      qkm2 = qkm1;
      qkm1 = qk;

      if( qk != 0 )
        r = pk/qk;
      if( r != 0 ) {
        t = Math.abs( (ans - r)/r );
        ans = r;
      }
      else
        t = 1.0;

      if( t < thresh )
        return ans;

      k1 += 1.0;
      k2 += 1.0;
      k3 += 2.0;
      k4 += 2.0;
      k5 += 1.0;
      k6 -= 1.0;
      k7 += 2.0;
      k8 += 2.0;

      if( (Math.abs(qk) + Math.abs(pk)) > big ) {
        pkm2 *= biginv;
        pkm1 *= biginv;
        qkm2 *= biginv;
        qkm1 *= biginv;
      }
      if( (Math.abs(qk) < biginv) || (Math.abs(pk) < biginv) ) {
        pkm2 *= big;
        pkm1 *= big;
        qkm2 *= big;
        qkm1 *= big;
      }
    }
    while( ++n < 300 );

    return ans;
  }

  /* Continued fraction expansion #2
   * for incomplete beta integral
   */

  private static double incbd( double a, double b, double x ) {
    double xk, pk, pkm1, pkm2, qk, qkm1, qkm2;
    double k1, k2, k3, k4, k5, k6, k7, k8;
    double r, t, ans, z, thresh;
    int n;

    k1 = a;
    k2 = b - 1.0;
    k3 = a;
    k4 = a + 1.0;
    k5 = 1.0;
    k6 = a + b;
    k7 = a + 1.0;;
    k8 = a + 2.0;

    pkm2 = 0.0;
    qkm2 = 1.0;
    pkm1 = 1.0;
    qkm1 = 1.0;
    z = x / (1.0-x);
    ans = 1.0;
    r = 1.0;
    n = 0;
    thresh = 3.0 * MACHEP;
    do {

      xk = -( z * k1 * k2 )/( k3 * k4 );
      pk = pkm1 +  pkm2 * xk;
      qk = qkm1 +  qkm2 * xk;
      pkm2 = pkm1;
      pkm1 = pk;
      qkm2 = qkm1;
      qkm1 = qk;

      xk = ( z * k5 * k6 )/( k7 * k8 );
      pk = pkm1 +  pkm2 * xk;
      qk = qkm1 +  qkm2 * xk;
      pkm2 = pkm1;
      pkm1 = pk;
      qkm2 = qkm1;
      qkm1 = qk;

      if( qk != 0 )
        r = pk/qk;
      if( r != 0 ) {
        t = Math.abs( (ans - r)/r );
        ans = r;
      }
      else
        t = 1.0;

      if( t < thresh )
        return ans;

      k1 += 1.0;
      k2 -= 1.0;
      k3 += 2.0;
      k4 += 2.0;
      k5 += 1.0;
      k6 += 1.0;
      k7 += 2.0;
      k8 += 2.0;

      if( (Math.abs(qk) + Math.abs(pk)) > big ) {
        pkm2 *= biginv;
        pkm1 *= biginv;
        qkm2 *= biginv;
        qkm1 *= biginv;
      }
      if( (Math.abs(qk) < biginv) || (Math.abs(pk) < biginv) ) {
        pkm2 *= big;
        pkm1 *= big;
        qkm2 *= big;
        qkm1 *= big;
      }
    }
    while( ++n < 300 );

    return ans;
  }

  /* Power series for incomplete beta integral.
     Use when b*x is small and x not too close to 1.  */

  private static double pseries( double a, double b, double x ) {
    double s, t, u, v, n, t1, z, ai;

    ai = 1.0 / a;
    u = (1.0 - b) * x;
    v = u / (a + 1.0);
    t1 = v;
    t = u;
    n = 2.0;
    s = 0.0;
    z = MACHEP * ai;
    while( Math.abs(v) > z ) {
      u = (n - b) * x / n;
      t *= u;
      v = t / (a + n);
      s += v;
      n += 1.0;
    }
    s += t1;
    s += ai;

    u = a * Math.log(x);
    if( (a+b) < MAXGAM && Math.abs(u) < MAXLOG ) {
      t = gamma(a+b)/(gamma(a)*gamma(b));
      s = s * t * Math.pow(x,a);
    }
    else {
      t = lgam(a+b) - lgam(a) - lgam(b) + u + Math.log(s);
      if( t < MINLOG )
        s = 0.0;
      else
        s = Math.exp(t);
      }
    return s;
  }

/*                                                      gamma.c
 *
 *      Gamma function
 *
 *
 *
 * SYNOPSIS:
 *
 * double x, y, gamma();
 * extern int sgngam;
 *
 * y = gamma( x );
 *
 *
 *
 * DESCRIPTION:
 *
 * Returns gamma function of the argument.  The result is
 * correctly signed, and the sign (+1 or -1) is also
 * returned in a global (extern) variable named sgngam.
 * This variable is also filled in by the logarithmic gamma
 * function lgam().
 *
 * Arguments |x| <= 34 are reduced by recurrence and the function
 * approximated by a rational function of degree 6/7 in the
 * interval (2,3).  Large arguments are handled by Stirling's
 * formula. Large negative arguments are made positive using
 * a reflection formula.
 *
 *
 * ACCURACY:
 *
 *                      Relative error:
 * arithmetic   domain     # trials      peak         rms
 *    DEC      -34, 34      10000       1.3e-16     2.5e-17
 *    IEEE    -170,-33      20000       2.3e-15     3.3e-16
 *    IEEE     -33,  33     20000       9.4e-16     2.2e-16
 *    IEEE      33, 171.6   20000       2.3e-15     3.2e-16
 *
 * Error for arguments outside the test range will be larger
 * owing to error amplification by the exponential function.
 *
 */

  public static double gamma(double x) {
    final double P[] = {
      1.60119522476751861407E-4d,
      1.19135147006586384913E-3d,
      1.04213797561761569935E-2d,
      4.76367800457137231464E-2d,
      2.07448227648435975150E-1d,
      4.94214826801497100753E-1d,
      9.99999999999999996796E-1d
    };
    final double Q[] = {
      -2.31581873324120129819E-5d,
       5.39605580493303397842E-4d,
      -4.45641913851797240494E-3d,
       1.18139785222060435552E-2d,
       3.58236398605498653373E-2d,
      -2.34591795718243348568E-1d,
       7.14304917030273074085E-2d,
       1.00000000000000000320E0d 
    };
    double p, q, z;
    int i;

    byte sgngam = 1;
    q = Math.abs(x);

    if( q > 33.0 ) {
      if( x < 0.0 ) {
        p = Math.floor(q);
        if( p == q )
          return( sgngam * INFINITY);
        i = (int) p;
        if( (i & 1) == 0 )
          sgngam = -1;
        z = q - p;
        if( z > 0.5 ) {
          p += 1.0;
          z = q - p;
        }
        z = q * Math.sin( PI * z );
        if( z == 0.0 ) {
          return( sgngam * INFINITY);
        }
        z = Math.abs(z);
        z = PI/(z * stirf(q) );
      }
      else {
        z = stirf(x);
      }
      return( sgngam * z );
    }

    z = 1.0;
    while( x >= 3.0 ) {
      x -= 1.0;
      z *= x;
    }

    while( x < 0.0 ) {
      if( x > -1.E-9 )
        return z/((1.0 + 0.5772156649015329 * x) * x);
      z /= x;
      x += 1.0;
    }

    while( x < 2.0 ) {
      if( x < 1.e-9 )
        return z/((1.0 + 0.5772156649015329 * x) * x);
      z /= x;
      x += 1.0;
    }

    if( (x == 2.0) || (x == 3.0) )
      return(z);

    x -= 2.0;
    p = polevl( x, P );
    q = polevl( x, Q );
    return( z * p / q );
  }

  /* Gamma function computed by Stirling's formula.
   * The polynomial STIR is valid for 33 <= x <= 172.
   */
  private static double stirf(double x) {
    /* Stirling's formula for the gamma function */
    final double STIR[] = {
       7.87311395793093628397E-4d,
      -2.29549961613378126380E-4d,
      -2.68132617805781232825E-3d,
       3.47222221605458667310E-3d,
       8.33333333333482257126E-2d,
    };
    final double MAXSTIR = 143.01608;
    double y, w, v;

    w = 1.0/x;
    w = 1.0 + w * polevl( w, STIR );
    y = Math.exp(x);
    if( x > MAXSTIR ) { /* Avoid overflow in pow() */
      v = Math.pow( x, 0.5 * x - 0.25 );
      y = v * (v / y);
    }
    else {
      y = Math.pow( x, x - 0.5 ) / y;
    }
    y = SQTPI * y * w;
    return y;
  }

/*                                                      lgam()
 *
 *      Natural logarithm of gamma function
 *
 *
 *
 * SYNOPSIS:
 *
 * double x, y, lgam();
 * extern int sgngam;
 *
 * y = lgam( x );
 *
 *
 *
 * DESCRIPTION:
 *
 * Returns the base e (2.718...) logarithm of the absolute
 * value of the gamma function of the argument.
 * The sign (+1 or -1) of the gamma function is returned in a
 * global (extern) variable named sgngam.
 *
 * For arguments greater than 13, the logarithm of the gamma
 * function is approximated by the logarithmic version of
 * Stirling's formula using a polynomial approximation of
 * degree 4. Arguments between -33 and +33 are reduced by
 * recurrence to the interval [2,3] of a rational approximation.
 * The cosecant reflection formula is employed for arguments
 * less than -33.
 *
 * Arguments greater than MAXLGM return MAXNUM and an error
 * message.  MAXLGM = 2.035093e36 for DEC
 * arithmetic or 2.556348e305 for IEEE arithmetic.
 *
 *
 *
 * ACCURACY:
 *
 *
 * arithmetic      domain        # trials     peak         rms
 *    DEC     0, 3                  7000     5.2e-17     1.3e-17
 *    DEC     2.718, 2.035e36       5000     3.9e-17     9.9e-18
 *    IEEE    0, 3                 28000     5.4e-16     1.1e-16
 *    IEEE    2.718, 2.556e305     40000     3.5e-16     8.3e-17
 * The error criterion was relative when the function magnitude
 * was greater than one but absolute when it was less than one.
 *
 * The following test used the relative error criterion, though
 * at certain points the relative error could be much higher than
 * indicated.
 *    IEEE    -200, -4             10000     4.8e-16     1.3e-16
 *
 */

/* Logarithm of gamma function */
  public static double lgam(double x) {
    /* A[]: Stirling's formula expansion of log gamma
     * B[], C[]: log gamma function between 2 and 3
     */
    final double A[] = {
       8.11614167470508450300E-4d,
      -5.95061904284301438324E-4d,
       7.93650340457716943945E-4d,
      -2.77777777730099687205E-3d,
       8.33333333333331927722E-2d
    };
    final double B[] = {
      -1.37825152569120859100E3d,
      -3.88016315134637840924E4d,
      -3.31612992738871184744E5d,
      -1.16237097492762307383E6d,
      -1.72173700820839662146E6d,
      -8.53555664245765465627E5d
    };
    final double C[] = {
      /* 1.00000000000000000000E0d, */
      -3.51815701436523470549E2d,
      -1.70642106651881159223E4d,
      -2.20528590553854454839E5d,
      -1.13933444367982507207E6d,
      -2.53252307177582951285E6d,
      -2.01889141433532773231E6d
    };
    final double LS2PI  =  0.91893853320467274178d;
    final double MAXLGM = 2.556348e305d;
    double p, q, w, z;
    int i;

    int sgngam = 1;

    if( x < -34.0 ) {
      q = -x;
      w = lgam(q); /* note this modifies sgngam! */
      p = Math.floor(q);
      if( p == q )
        return( sgngam * INFINITY );
      i = (int) p;
      if( (i & 1) == 0 )
        sgngam = -1;
      else
        sgngam = 1;
      z = q - p;
      if( z > 0.5 ) {
        p += 1.0;
        z = p - q;
      }
      z = q * Math.sin( PI * z );
      if( z == 0.0 )
        return( sgngam * INFINITY );
      z = LOGPI - Math.log( z ) - w;
      return( z );
    }

    if( x < 13.0 ) {
      z = 1.0;
      while( x >= 3.0 ) {
        x -= 1.0;
        z *= x;
      }
      while( x < 2.0 ) {
        if( x == 0.0 )
          return( sgngam * INFINITY );
        z /= x;
        x += 1.0;
      }
      if( z < 0.0 ) {
        sgngam = -1;
        z = -z;
      }
      else
        sgngam = 1;
      if( x == 2.0 )
        return( Math.log(z) );
      x -= 2.0;
      p = x * polevl( x, B ) / p1evl( x, C);
      return( Math.log(z) + p );
    }

    if( x > MAXLGM ) {
      return( sgngam * INFINITY );
    }

    q = ( x - 0.5 ) * Math.log(x) - x + LS2PI;
    if( x > 1.0e8 )
      return( q );

    p = 1.0/(x*x);
    if( x >= 1000.0 )
      q += ((   7.9365079365079365079365e-4 * p
                - 2.7777777777777777777778e-3) *p
                + 0.0833333333333333333333) / x;
    else
      q += polevl( p, A ) / x;
    return( q );
  }

/*                                                      incbi()
 *
 *      Inverse of imcomplete beta integral
 *
 *
 *
 * SYNOPSIS:
 *
 * double a, b, x, y, incbi();
 *
 * x = incbi( a, b, y );
 *
 *
 *
 * DESCRIPTION:
 *
 * Given y, the function finds x such that
 *
 *  incbet( a, b, x ) = y .
 *
 * The routine performs interval halving or Newton iterations to find the
 * root of incbet(a,b,x) - y = 0.
 *
 *
 * ACCURACY:
 *
 *                      Relative error:
 *                x     a,b
 * arithmetic   domain  domain  # trials    peak       rms
 *    IEEE      0,1    .5,10000   50000    5.8e-12   1.3e-13
 *    IEEE      0,1   .25,100    100000    1.8e-13   3.9e-15
 *    IEEE      0,1     0,5       50000    1.1e-12   5.5e-15
 *    VAX       0,1    .5,100     25000    3.5e-14   1.1e-15
 * With a and b constrained to half-integer or integer values:
 *    IEEE      0,1    .5,10000   50000    5.8e-12   1.1e-13
 *    IEEE      0,1    .5,100    100000    1.7e-14   7.9e-16
 * With a = .5, b constrained to half-integer or integer values:
 *    IEEE      0,1    .5,10000   10000    8.3e-11   1.0e-11
 */
  public static double incbi( double aa, double bb, double yy0 )
         throws MyArithmeticException {
    double a, b, y0, d, y, x, x0, x1, lgm, yp, di, dithresh, yl, yh, xt;
    int i, dir;
    boolean nflg, rflg, ihalve=true;


    i = 0;
    if( yy0 <= 0 )
      return(0.0);
    if( yy0 >= 1.0 )
      return(1.0);
    x0 = 0.0;
    yl = 0.0;
    x1 = 1.0;
    yh = 1.0;
    nflg = false;

    if( aa <= 1.0 || bb <= 1.0 ) {
      dithresh = 1.0e-6;
      rflg = false;
      a = aa;
      b = bb;
      y0 = yy0;
      x = a/(a+b);
      y = incbet( a, b, x );
      // goto ihalve;
      // ihalve = true and the else is not ended as it
      // should be so from here it goes directly to the while
    }
    else {
      dithresh = 1.0e-4;
    // } Changed so that goto ihalve (above could be simulated)
    /* approximation to inverse function */

    yp = -ndtri(yy0);

    if( yy0 > 0.5 ) {
      rflg = true;
      a = bb;
      b = aa;
      y0 = 1.0 - yy0;
      yp = -yp;
    }
    else {
      rflg = false;
      a = aa;
      b = bb;
      y0 = yy0;
    }

    lgm = (yp * yp - 3.0)/6.0;
    x = 2.0/( 1.0/(2.0*a-1.0)  +  1.0/(2.0*b-1.0) );
    d = yp * Math.sqrt( x + lgm ) / x
        - ( 1.0/(2.0*b-1.0) - 1.0/(2.0*a-1.0) )
        * (lgm + 5.0/6.0 - 2.0/(3.0*x));
    d = 2.0 * d;
    if( d < MINLOG ) {
      x = 1.0;
      throw new MyArithmeticException();
    }
    x = a/( a + b * Math.exp(d) );
    y = incbet( a, b, x );
    yp = (y - y0)/y0;
    if( Math.abs(yp) < 0.2 ) {
      ihalve=false;
    }
    } // the } that was previously commented is closed here
    while(true) {
      if(ihalve) {
        /* Resort to interval halving if not close enough. */

        dir = 0;
        di = 0.5;
        for( i=0; i<100; i++ ) {
          if( i != 0 ) {
            x = x0  +  di * (x1 - x0);
            if( x == 1.0 )
              x = 1.0 - MACHEP;
            if( x == 0.0 ) {
              di = 0.5;
              x = x0  +  di * (x1 - x0);
              if( x == 0.0 )
                throw new MyArithmeticException();
            }
            y = incbet( a, b, x );
            yp = (x1 - x0)/(x1 + x0);
            if( Math.abs(yp) < dithresh ) {
              ihalve=false;
              continue;
            }
            yp = (y-y0)/y0;
            if( Math.abs(yp) < dithresh ) {
              ihalve=false;
              continue;
            }
          }
          if( y < y0 ) {
            x0 = x;
            yl = y;
            if( dir < 0 ) {
              dir = 0;
              di = 0.5;
            }
            else if( dir > 3 )
              di = 1.0 - (1.0 - di) * (1.0 - di);
            else if( dir > 1 )
              di = 0.5 * di + 0.5;
            else
              di = (y0 - y)/(yh - yl);
            dir += 1;
            if( x0 > 0.75 ) {
              if( rflg ) {
                rflg = false;
                a = aa;
                b = bb;
                y0 = yy0;
              }
              else {
                rflg = true;
                a = bb;
                b = aa;
                y0 = 1.0 - yy0;
              }
              x = 1.0 - x;
              y = incbet( a, b, x );
              x0 = 0.0;
              yl = 0.0;
              x1 = 1.0;
              yh = 1.0;
              ihalve=true;
              continue;
            }
          }
          else {
            x1 = x;
            if( rflg && x1 < MACHEP ) {
              x = 0.0;
              return incbiDone(rflg,x);
            }
            yh = y;
            if( dir > 0 ) {
              dir = 0;
              di = 0.5;
            }
            else if( dir < -3 )
              di = di * di;
            else if( dir < -1 )
              di = 0.5 * di;
            else
              di = (y - y0)/(yh - yl);
            dir -= 1;
          }
        }
        if( x0 >= 1.0 ) {
          x = 1.0 - MACHEP;
          return incbiDone(rflg,x);
        }
        if( x <= 0.0 ) {
          throw new MyArithmeticException();
        }
      } 
      else { // newt
        if( nflg )
          return incbiDone(rflg,x);
        nflg = true;
        lgm = lgam(a+b) - lgam(a) - lgam(b);

        for( i=0; i<8; i++ ) {
          /* Compute the function at this point. */
          if( i != 0 )
            y = incbet(a,b,x);
          if( y < yl ) {
            x = x0;
            y = yl;
          }
          else if( y > yh ) {
            x = x1;
            y = yh;
          }
          else if( y < y0 ) {
            x0 = x;
            yl = y;
          }
          else {
            x1 = x;
            yh = y;
          }
          if( x == 1.0 || x == 0.0 )
            break;
          /* Compute the derivative of the function at this point. */
          d = (a - 1.0) * Math.log(x) + (b - 1.0) * Math.log(1.0-x) + lgm;
          if( d < MINLOG )
            return incbiDone(rflg,x);
          if( d > MAXLOG )
            break;
          d = Math.exp(d);
          /* Compute the step to the next approximation of x. */
          d = (y - y0)/d;
          xt = x - d;
          if( xt <= x0 ) {
            y = (x - x0) / (x1 - x0);
            xt = x0 + 0.5 * y * (x - x0);
            if( xt <= 0.0 )
              break;
          }
          if( xt >= x1 ) {
            y = (x1 - x) / (x1 - x0);
            xt = x1 - 0.5 * y * (x1 - x);
            if( xt >= 1.0 )
              break;
          }
          x = xt;
          if( Math.abs(d/x) < 128.0 * MACHEP )
            return incbiDone(rflg,x);
        }
        /* Did not converge.  */
        dithresh = 256.0 * MACHEP;
        ihalve=true;
        continue;
      } // end if(ihalve)
    } // end for(boolean ihalve;true;)
  }

  private static double incbiDone(boolean rflg, double x) {
    if( rflg ) {
      if( x <= MACHEP )
        x = 1.0 - MACHEP;
      else
        x = 1.0 - x;
      }
    return( x );
  }

/*                                                      igam.c
 *
 *      Incomplete gamma integral
 *
 *
 *
 * SYNOPSIS:
 *
 * double a, x, y, igam();
 *
 * y = igam( a, x );
 *
 * DESCRIPTION:
 *
 * The function is defined by
 *
 *                           x
 *                            -
 *                   1       | |  -t  a-1
 *  igam(a,x)  =   -----     |   e   t   dt.
 *                  -      | |
 *                 | (a)    -
 *                           0
 *
 *
 * In this implementation both arguments must be positive.
 * The integral is evaluated by either a power series or
 * continued fraction expansion, depending on the relative
 * values of a and x.
 *
 * ACCURACY:
 *
 *                      Relative error:
 * arithmetic   domain     # trials      peak         rms
 *    IEEE      0,30       200000       3.6e-14     2.9e-15
 *    IEEE      0,100      300000       9.9e-14     1.5e-14
 */

/* left tail of incomplete gamma function:
 *
 *          inf.      k
 *   a  -x   -       x
 *  x  e     >   ----------
 *           -     -
 *          k=0   | (a+k+1)
 *
 */

  public static double igam( double a, double x )
         throws MyArithmeticException {
    double ans, ax, c, r;

    if( (x <= 0) || ( a <= 0) )
      return( 0.0 );

    if( (x > 1.0) && (x > a ) )
      return( 1.0 - igamc(a,x) );

    /* Compute  x**a * exp(-x) / gamma(a)  */
    ax = a * Math.log(x) - x - lgam(a);
    if( ax < -MAXLOG ) {
      throw new MyArithmeticException();
    }
    ax = Math.exp(ax);

    /* power series */
    r = a;
    c = 1.0;
    ans = 1.0;

    do {
      r += 1.0;
      c *= x/r;
      ans += c;
    }
    while( c/ans > MACHEP );

    return( ans * ax/a );
  }

/*                                                      igamc()
 *
 *      Complemented incomplete gamma integral
 *
 *
 *
 * SYNOPSIS:
 *
 * double a, x, y, igamc();
 *
 * y = igamc( a, x );
 *
 * DESCRIPTION:
 *
 * The function is defined by
 *
 *
 *  igamc(a,x)   =   1 - igam(a,x)
 *
 *                            inf.
 *                              -
 *                     1       | |  -t  a-1
 *               =   -----     |   e   t   dt.
 *                    -      | |
 *                   | (a)    -
 *                             x
 *
 *
 * In this implementation both arguments must be positive.
 * The integral is evaluated by either a power series or
 * continued fraction expansion, depending on the relative
 * values of a and x.
 *
 * ACCURACY:
 *
 * Tested at random a, x.
 *                a         x                      Relative error:
 * arithmetic   domain   domain     # trials      peak         rms
 *    IEEE     0.5,100   0,100      200000       1.9e-14     1.7e-15
 *    IEEE     0.01,0.5  0,100      200000       1.4e-13     1.6e-15
 */


  public static double igamc( double a, double x )
         throws MyArithmeticException {
    double ans, ax, c, yc, r, t, y, z;
    double pk, pkm1, pkm2, qk, qkm1, qkm2;

    if( (x <= 0) || ( a <= 0) )
      return( 1.0 );

    if( (x < 1.0) || (x < a) )
      return( 1.0 - igam(a,x) );

    ax = a * Math.log(x) - x - lgam(a);
    if( ax < -MAXLOG ) {
      throw new MyArithmeticException();
    }
    ax = Math.exp(ax);

    /* continued fraction */
    y = 1.0 - a;
    z = x + y + 1.0;
    c = 0.0;
    pkm2 = 1.0;
    qkm2 = x;
    pkm1 = x + 1.0;
    qkm1 = z * x;
    ans = pkm1/qkm1;

    do {
      c += 1.0;
      y += 1.0;
      z += 2.0;
      yc = y * c;
      pk = pkm1 * z  -  pkm2 * yc;
      qk = qkm1 * z  -  qkm2 * yc;
      if( qk != 0 ) {
        r = pk/qk;
        t = Math.abs( (ans - r)/r );
        ans = r;
      }
      else
        t = 1.0;
      pkm2 = pkm1;
      pkm1 = pk;
      qkm2 = qkm1;
      qkm1 = qk;
      if( Math.abs(pk) > big ) {
        pkm2 *= biginv;
        pkm1 *= biginv;
        qkm2 *= biginv;
        qkm1 *= biginv;
      }
    }
    while( t > MACHEP );

    return( ans * ax );
  }

/*                                                      chdtr.c
 *
 *      Chi-square distribution
 *
 *
 *
 * SYNOPSIS:
 *
 * double df, x, y, chdtr();
 *
 * y = chdtr( df, x );
 *
 *
 *
 * DESCRIPTION:
 *
 * Returns the area under the left hand tail (from 0 to x)
 * of the Chi square probability density function with
 * v degrees of freedom.
 *
 *
 *                                  inf.
 *                                    -
 *                        1          | |  v/2-1  -t/2
 *  P( x | v )   =   -----------     |   t      e     dt
 *                    v/2  -       | |
 *                   2    | (v/2)   -
 *                                   x
 *
 * where x is the Chi-square variable.
 *
 * The incomplete gamma integral is used, according to the
 * formula
 *
 *      y = chdtr( v, x ) = igam( v/2.0, x/2.0 ).
 *
 *
 * The arguments must both be positive.
 *
 *
 *
 * ACCURACY:
 *
 * See igam().
 *
 * ERROR MESSAGES:
 *
 *   message         condition      value returned
 * chdtr domain   x < 0 or v < 1        0.0
 */

  public static double chdtr(double df,double x)
         throws MyArithmeticException {

    if( (x < 0.0) || (df < 1.0) ) {
      throw new MyArithmeticException();
    }
    return( igam( df/2.0, x/2.0 ) );
  }

/*                                                      chdtrc()
 *
 *      Complemented Chi-square distribution
 *
 *
 *
 * SYNOPSIS:
 *
 * double v, x, y, chdtrc();
 *
 * y = chdtrc( v, x );
 *
 *
 *
 * DESCRIPTION:
 *
 * Returns the area under the right hand tail (from x to
 * infinity) of the Chi square probability density function
 * with v degrees of freedom:
 *
 *
 *                                  inf.
 *                                    -
 *                        1          | |  v/2-1  -t/2
 *  P( x | v )   =   -----------     |   t      e     dt
 *                    v/2  -       | |
 *                   2    | (v/2)   -
 *                                   x
 *
 * where x is the Chi-square variable.
 *
 * The incomplete gamma integral is used, according to the
 * formula
 *
 *      y = chdtr( v, x ) = igamc( v/2.0, x/2.0 ).
 *
 *
 * The arguments must both be positive.
 *
 *
 *
 * ACCURACY:
 *
 * See igamc().
 *
 * ERROR MESSAGES:
 *
 *   message         condition      value returned
 * chdtrc domain  x < 0 or v < 1        0.0
 */
  public static double chdtrc(double df,double x)
         throws MyArithmeticException {

    if( (x < 0.0) || (df < 1.0) ) {
      throw new MyArithmeticException();
    }
    return( igamc( df/2.0, x/2.0 ) );
  }

/*                                                      chdtri()
 *
 *      Inverse of complemented Chi-square distribution
 *
 *
 *
 * SYNOPSIS:
 *
 * double df, x, y, chdtri();
 *
 * x = chdtri( df, y );
 *
 *
 *
 *
 * DESCRIPTION:
 *
 * Finds the Chi-square argument x such that the integral
 * from x to infinity of the Chi-square density is equal
 * to the given cumulative probability y.
 *
 * This is accomplished using the inverse gamma integral
 * function and the relation
 *
 *    x/2 = igami( df/2, y );
 *
 *
 *
 *
 * ACCURACY:
 *
 * See igami.c.
 *
 * ERROR MESSAGES:
 *
 *   message         condition      value returned
 * chdtri domain   y < 0 or y > 1        0.0
 *                     v < 1
 *
 */

  public static double chdtri( double df, double y )
         throws MyArithmeticException {

    if( (y < 0.0) || (y > 1.0) || (df < 1.0) ) {
      throw new MyArithmeticException();
    }

    return( 2.0 * igami( 0.5 * df, y ) );
  }

/*                                                      igami()
 *
 *      Inverse of complemented imcomplete gamma integral
 *
 *
 *
 * SYNOPSIS:
 *
 * double a, x, p, igami();
 *
 * x = igami( a, p );
 *
 * DESCRIPTION:
 *
 * Given p, the function finds x such that
 *
 *  igamc( a, x ) = p.
 *
 * Starting with the approximate value
 *
 *         3
 *  x = a t
 *
 *  where
 *
 *  t = 1 - d - ndtri(p) sqrt(d)
 *
 * and
 *
 *  d = 1/9a,
 *
 * the routine performs up to 10 Newton iterations to find the
 * root of igamc(a,x) - p = 0.
 *
 * ACCURACY:
 *
 * Tested at random a, p in the intervals indicated.
 *
 *                a        p                      Relative error:
 * arithmetic   domain   domain     # trials      peak         rms
 *    IEEE     0.5,100   0,0.5       100000       1.0e-14     1.7e-15
 *    IEEE     0.01,0.5  0,0.5       100000       9.0e-14     3.4e-15
 *    IEEE    0.5,10000  0,0.5        20000       2.3e-13     3.8e-14
 */

  public static double igami( double a, double y0 )
         throws MyArithmeticException {
    double x0, x1, x, yl, yh, y, d, lgm, dithresh;
    int i, dir;

    /* bound the solution */
    x0 = MAXNUM;
    yl = 0;
    x1 = 0;
    yh = 1.0;
    dithresh = 5.0 * MACHEP;

    /* approximation to inverse function */
    d = 1.0/(9.0*a);
    y = ( 1.0 - d - ndtri(y0) * Math.sqrt(d) );
    x = a * y * y * y;

    lgm = lgam(a);

    for( i=0; i<10; i++ ) {
      if( x > x0 || x < x1 )
        break;
      y = igamc(a,x);
      if( y < yl || y > yh )
        break;
      if( y < y0 ) {
        x0 = x;
        yl = y;
      }
      else {
        x1 = x;
        yh = y;
      }
/* compute the derivative of the function at this point */
      d = (a - 1.0) * Math.log(x) - x - lgm;
      if( d < -MAXLOG )
        break;
      d = -Math.exp(d);
/* compute the step to the next approximation of x */
      d = (y - y0)/d;
      if( Math.abs(d/x) < MACHEP )
        return x;
      x = x - d;
    }

/* Resort to interval halving if Newton iteration did not converge. */

    d = 0.0625;
    if( x0 == MAXNUM ) {
      if( x <= 0.0 )
        x = 1.0;
      while( x0 == MAXNUM ) {
        x = (1.0 + d) * x;
        y = igamc( a, x );
        if( y < y0 ) {
          x0 = x;
          yl = y;
          break;
        }
        d = d + d;
      }
    }
    d = 0.5;
    dir = 0;

    for( i=0; i<400; i++ ) {
      x = x1  +  d * (x0 - x1);
      y = igamc( a, x );
      lgm = (x0 - x1)/(x1 + x0);
      if( Math.abs(lgm) < dithresh )
        break;
      lgm = (y - y0)/y0;
      if( Math.abs(lgm) < dithresh )
        break;
      if( x <= 0.0 )
        break;
      if( y >= y0 ) {
        x1 = x;
        yh = y;
        if( dir < 0 ) {
          dir = 0;
          d = 0.5;
        }
        else if( dir > 1 )
          d = 0.5 * d + 0.5;
        else
          d = (y0 - yl)/(yh - yl);
        dir += 1;
      }
      else {
        x0 = x;
        yl = y;
        if( dir > 0 ) {
          dir = 0;
          d = 0.5;
        }
        else if( dir < -1 )
          d = 0.5 * d;
        else
          d = (y0 - yl)/(yh - yl);
        dir -= 1;
      }
    }
    if( x == 0.0 )
      throw new MyArithmeticException();

    return( x );
  }

/*                                                      fdtr.c
 *
 *      F distribution
 *
 *
 *
 * SYNOPSIS:
 *
 * int df1, df2;
 * double x, y, fdtr();
 *
 * y = fdtr( df1, df2, x );
 *
 * DESCRIPTION:
 *
 * Returns the area from zero to x under the F density
 * function (also known as Snedcor's density or the
 * variance ratio density).  This is the density
 * of x = (u1/df1)/(u2/df2), where u1 and u2 are random
 * variables having Chi square distributions with df1
 * and df2 degrees of freedom, respectively.
 *
 * The incomplete beta integral is used, according to the
 * formula
 *
 *      P(x) = incbet( df1/2, df2/2, (df1*x/(df2 + df1*x) ).
 *
 *
 * The arguments a and b are greater than zero, and x is
 * nonnegative.
 *
 * ACCURACY:
 *
 * Tested at random points (a,b,x).
 *
 *                x     a,b                     Relative error:
 * arithmetic  domain  domain     # trials      peak         rms
 *    IEEE      0,1    0,100       100000      9.8e-15     1.7e-15
 *    IEEE      1,5    0,100       100000      6.5e-15     3.5e-16
 *    IEEE      0,1    1,10000     100000      2.2e-11     3.3e-12
 *    IEEE      1,5    1,10000     100000      1.1e-11     1.7e-13
 * See also incbet.c.
 *
 *
 * ERROR MESSAGES:
 *
 *   message         condition      value returned
 * fdtr domain     a<0, b<0, x<0         0.0
 *
 */

  public static double fdtr( int ia, int ib, double x )
         throws MyArithmeticException {
    double a, b, w;

    if( (ia < 1) || (ib < 1) || (x < 0.0) ) {
      throw new MyArithmeticException();
    }
    a = ia;
    b = ib;
    w = a * x;
    w = w / (b + w);
    return( incbet(0.5*a, 0.5*b, w) );
  }

/*                                                      fdtrc()
 *
 *      Complemented F distribution
 *
 *
 *
 * SYNOPSIS:
 *
 * int df1, df2;
 * double x, y, fdtrc();
 *
 * y = fdtrc( df1, df2, x );
 *
 * DESCRIPTION:
 *
 * Returns the area from x to infinity under the F density
 * function (also known as Snedcor's density or the
 * variance ratio density).
 *
 *
 *                      inf.
 *                       -
 *              1       | |  a-1      b-1
 * 1-P(x)  =  ------    |   t    (1-t)    dt
 *            B(a,b)  | |
 *                     -
 *                      x
 *
 *
 * The incomplete beta integral is used, according to the
 * formula
 *
 *      P(x) = incbet( df2/2, df1/2, (df2/(df2 + df1*x) ).
 *
 *
 * ACCURACY:
 *
 * Tested at random points (a,b,x) in the indicated intervals.
 *                x     a,b                     Relative error:
 * arithmetic  domain  domain     # trials      peak         rms
 *    IEEE      0,1    1,100       100000      3.7e-14     5.9e-16
 *    IEEE      1,5    1,100       100000      8.0e-15     1.6e-15
 *    IEEE      0,1    1,10000     100000      1.8e-11     3.5e-13
 *    IEEE      1,5    1,10000     100000      2.0e-11     3.0e-12
 * See also incbet.c.
 *
 * ERROR MESSAGES:
 *
 *   message         condition      value returned
 * fdtrc domain    a<0, b<0, x<0         0.0
 *
 */

  public static double fdtrc( int ia, int ib, double x )
         throws MyArithmeticException {
    double a, b, w;

    if( (ia < 1) || (ib < 1) || (x < 0.0) ) {
      throw new MyArithmeticException();
    }

    a = ia;
    b = ib;
    w = b / (b + a * x);
    return( incbet( 0.5*b, 0.5*a, w ) );
  }

  
/*                                                      fdtri()
 *
 *      Inverse of complemented F distribution
 *
 *
 *
 * SYNOPSIS:
 *
 * int df1, df2;
 * double x, p, fdtri();
 *
 * x = fdtri( df1, df2, p );
 *
 * DESCRIPTION:
 *
 * Finds the F density argument x such that the integral
 * from x to infinity of the F density is equal to the
 * given probability p.
 *
 * This is accomplished using the inverse beta integral
 * function and the relations
 *
 *      z = incbi( df2/2, df1/2, p )
 *      x = df2 (1-z) / (df1 z).
 *
 * Note: the following relations hold for the inverse of
 * the uncomplemented F distribution:
 *
 *      z = incbi( df1/2, df2/2, p )
 *      x = df2 z / (df1 (1-z)).
 *
 * ACCURACY:
 *
 * Tested at random points (a,b,p).
 *
 *              a,b                     Relative error:
 * arithmetic  domain     # trials      peak         rms
 *  For p between .001 and 1:
 *    IEEE     1,100       100000      8.3e-15     4.7e-16
 *    IEEE     1,10000     100000      2.1e-11     1.4e-13
 *  For p between 10^-6 and 10^-3:
 *    IEEE     1,100        50000      1.3e-12     8.4e-15
 *    IEEE     1,10000      50000      3.0e-12     4.8e-14
 * See also fdtrc.c.
 *
 * ERROR MESSAGES:
 *
 *   message         condition      value returned
 * fdtri domain   p <= 0 or p > 1       0.0
 *                     v < 1
 *
 */
  public static double fdtri( int ia, int ib, double y )
         throws MyArithmeticException {
    double a, b, w, x;

    if( (ia < 1) || (ib < 1) || (y <= 0.0) || (y > 1.0) ) {
      throw new MyArithmeticException();
    }
    a = ia;
    b = ib;
    /* Compute probability for x = 0.5.  */
    w = incbet( 0.5*b, 0.5*a, 0.5 );
    /* If that is greater than y, then the solution w < .5.
       Otherwise, solve at 1-y to remove cancellation in (b - b*w).  */
    if( w > y || y < 0.001) {
      w = incbi( 0.5*b, 0.5*a, y );
      x = (b - b*w)/(a*w);
    }
    else {
      w = incbi( 0.5*a, 0.5*b, 1.0-y );
      x = b*w/(a*(1.0-w));
    }
    return(x);
  }

/*                                                      gdtr.c
 *
 *      Gamma distribution function
 *
 *
 *
 * SYNOPSIS:
 *
 * double a, b, x, y, gdtr();
 *
 * y = gdtr( a, b, x );
 *
 *
 *
 * DESCRIPTION:
 *
 * Returns the integral from zero to x of the gamma probability
 * density function:
 *
 *
 *                x
 *        b       -
 *       a       | |   b-1  -at
 * y =  -----    |    t    e    dt
 *       -     | |
 *      | (b)   -
 *               0
 *
 *  The incomplete gamma integral is used, according to the
 * relation
 *
 * y = igam( b, ax ).
 *
 *
 * ACCURACY:
 *
 * See igam().
 *
 * ERROR MESSAGES:
 *
 *   message         condition      value returned
 * gdtr domain         x < 0            0.0
 *
 */

  public static double gdtr( double a, double b, double x )
         throws MyArithmeticException {

    if( x < 0.0 ) {
      throw new MyArithmeticException();
    }
    return(  igam( b, a * x )  );
  }

/*                                                      gdtrc.c
 *
 *      Complemented gamma distribution function
 *
 *
 *
 * SYNOPSIS:
 *
 * double a, b, x, y, gdtrc();
 *
 * y = gdtrc( a, b, x );
 *
 *
 *
 * DESCRIPTION:
 *
 * Returns the integral from x to infinity of the gamma
 * probability density function:
 *
 *
 *               inf.
 *        b       -
 *       a       | |   b-1  -at
 * y =  -----    |    t    e    dt
 *       -     | |
 *      | (b)   -
 *               x
 *
 *  The incomplete gamma integral is used, according to the
 * relation
 *
 * y = igamc( b, ax ).
 *
 *
 * ACCURACY:
 *
 * See igamc().
 *
 * ERROR MESSAGES:
 *
 *   message         condition      value returned
 * gdtrc domain         x < 0            0.0
 *
 */
  public static double gdtrc( double a, double b, double x )
         throws MyArithmeticException {

    if( x < 0.0 ) {
      throw new MyArithmeticException();
    }
    return(  igamc( b, a * x )  );
  }

/*                                                      pdtr.c
 *
 *      Poisson distribution
 *
 *
 *
 * SYNOPSIS:
 *
 * int k;
 * double m, y, pdtr();
 *
 * y = pdtr( k, m );
 *
 *
 *
 * DESCRIPTION:
 *
 * Returns the sum of the first k terms of the Poisson
 * distribution:
 *
 *   k         j
 *   --   -m  m
 *   >   e    --
 *   --       j!
 *  j=0
 *
 * The terms are not summed directly; instead the incomplete
 * gamma integral is employed, according to the relation
 *
 * y = pdtr( k, m ) = igamc( k+1, m ).
 *
 * The arguments must both be positive.
 *
 *
 *
 * ACCURACY:
 *
 * See igamc().
 *
 */
  public static double pdtr( int k, double m )
         throws MyArithmeticException {
    if( (k < 0) || (m <= 0.0) )
      throw new MyArithmeticException();
    return( igamc( k+1, m ) );
  }

/*                                                      pdtrc()
 *
 *      Complemented poisson distribution
 *
 *
 *
 * SYNOPSIS:
 *
 * int k;
 * double m, y, pdtrc();
 *
 * y = pdtrc( k, m );
 *
 *
 *
 * DESCRIPTION:
 *
 * Returns the sum of the terms k+1 to infinity of the Poisson
 * distribution:
 *
 *  inf.       j
 *   --   -m  m
 *   >   e    --
 *   --       j!
 *  j=k+1
 *
 * The terms are not summed directly; instead the incomplete
 * gamma integral is employed, according to the formula
 *
 * y = pdtrc( k, m ) = igam( k+1, m ).
 *
 * The arguments must both be positive.
 *
 *
 *
 * ACCURACY:
 *
 * See igam.c.
 *
 */
  public static double pdtrc( int k, double m )
         throws MyArithmeticException {

    if( (k < 0) || (m <= 0.0) )
      throw new MyArithmeticException();
    return( igam( k+1, m ) );
  }

/*                                                      pdtri()
 *
 *      Inverse Poisson distribution
 *
 *
 *
 * SYNOPSIS:
 *
 * int k;
 * double m, y, pdtr();
 *
 * m = pdtri( k, y );
 *
 *
 *
 *
 * DESCRIPTION:
 *
 * Finds the Poisson variable x such that the integral
 * from 0 to x of the Poisson density is equal to the
 * given probability y.
 *
 * This is accomplished using the inverse gamma integral
 * function and the relation
 *
 *    m = igami( k+1, y ).
 *
 *
 *
 *
 * ACCURACY:
 *
 * See igami.c.
 *
 * ERROR MESSAGES:
 *
 *   message         condition      value returned
 * pdtri domain    y < 0 or y >= 1       0.0
 *                     k < 0
 *
 */
  public static double pdtri( int k, double y )
         throws MyArithmeticException {

    if( (k < 0) || (y < 0.0) || (y >= 1.0) )
      throw new MyArithmeticException();
    return igami( k+1, y );
  }

/*                                                      bdtr.c
 *
 *      Binomial distribution
 *
 *
 *
 * SYNOPSIS:
 *
 * int k, n;
 * double p, y, bdtr();
 *
 * y = bdtr( k, n, p );
 *
 * DESCRIPTION:
 *
 * Returns the sum of the terms 0 through k of the Binomial
 * probability density:
 *
 *   k
 *   --  ( n )   j      n-j
 *   >   (   )  p  (1-p)
 *   --  ( j )
 *  j=0
 *
 * The terms are not summed directly; instead the incomplete
 * beta integral is employed, according to the formula
 *
 * y = bdtr( k, n, p ) = incbet( n-k, k+1, 1-p ).
 *
 * The arguments must be positive, with p ranging from 0 to 1.
 *
 * ACCURACY:
 *
 * Tested at random points (a,b,p), with p between 0 and 1.
 *
 *               a,b                     Relative error:
 * arithmetic  domain     # trials      peak         rms
 *  For p between 0.001 and 1:
 *    IEEE     0,100       100000      4.3e-15     2.6e-16
 * See also incbet.c.
 *
 * ERROR MESSAGES:
 *
 *   message         condition      value returned
 * bdtr domain         k < 0            0.0
 *                     n < k
 *                     x < 0, x > 1
 */

  public static double bdtr( int k, int n, double p )
         throws MyArithmeticException {
    double dk, dn;

    if( (p < 0.0) || (p > 1.0) || (k < 0) || (n < k))
      throw new MyArithmeticException();
    if( k == n )
      return( 1.0 );

    dn = n - k;
    if( k == 0 ) {
      dk = Math.pow( 1.0-p, dn );
    }
    else {
      dk = k + 1;
      dk = incbet( dn, dk, 1.0 - p );
    }
    return( dk );
  }

/*                                                      bdtrc()
 *
 *      Complemented binomial distribution
 *
 *
 *
 * SYNOPSIS:
 *
 * int k, n;
 * double p, y, bdtrc();
 *
 * y = bdtrc( k, n, p );
 *
 * DESCRIPTION:
 *
 * Returns the sum of the terms k+1 through n of the Binomial
 * probability density:
 *
 *   n
 *   --  ( n )   j      n-j
 *   >   (   )  p  (1-p)
 *   --  ( j )
 *  j=k+1
 *
 * The terms are not summed directly; instead the incomplete
 * beta integral is employed, according to the formula
 *
 * y = bdtrc( k, n, p ) = incbet( k+1, n-k, p ).
 *
 * The arguments must be positive, with p ranging from 0 to 1.
 *
 * ACCURACY:
 *
 * Tested at random points (a,b,p).
 *
 *               a,b                     Relative error:
 * arithmetic  domain     # trials      peak         rms
 *  For p between 0.001 and 1:
 *    IEEE     0,100       100000      6.7e-15     8.2e-16
 *  For p between 0 and .001:
 *    IEEE     0,100       100000      1.5e-13     2.7e-15
 *
 * ERROR MESSAGES:
 *
 *   message         condition      value returned
 * bdtrc domain      x<0, x>1, n<k       0.0
 */

  public static double bdtrc( int k, int n, double p )
         throws MyArithmeticException {
    double dk, dn;

    if( (p < 0.0) || (p > 1.0))
      throw new MyArithmeticException();
    if( k < 0 )
      return( 1.0 );

    if( n < k ) {
      throw new MyArithmeticException();
    }

    if( k == n )
      return( 0.0 );
    dn = n - k;
    if( k == 0 ) {
      if( p < .01 )
        dk = -expm1( dn * log1p(-p) );
      else
        dk = 1.0 - Math.pow( 1.0-p, dn );
    }
    else {
      dk = k + 1;
      dk = incbet( dk, dn, p );
    }
    return( dk );
  }

/*                                                      bdtri()
 *
 *      Inverse binomial distribution
 *
 *
 *
 * SYNOPSIS:
 *
 * int k, n;
 * double p, y, bdtri();
 *
 * p = bdtr( k, n, y );
 *
 * DESCRIPTION:
 *
 * Finds the event probability p such that the sum of the
 * terms 0 through k of the Binomial probability density
 * is equal to the given cumulative probability y.
 *
 * This is accomplished using the inverse beta integral
 * function and the relation
 *
 * 1 - p = incbi( n-k, k+1, y ).
 *
 * ACCURACY:
 *
 * Tested at random points (a,b,p).
 *
 *               a,b                     Relative error:
 * arithmetic  domain     # trials      peak         rms
 *  For p between 0.001 and 1:
 *    IEEE     0,100       100000      2.3e-14     6.4e-16
 *    IEEE     0,10000     100000      6.6e-12     1.2e-13
 *  For p between 10^-6 and 0.001:
 *    IEEE     0,100       100000      2.0e-12     1.3e-14
 *    IEEE     0,10000     100000      1.5e-12     3.2e-14
 * See also incbi.c.
 *
 * ERROR MESSAGES:
 *
 *   message         condition      value returned
 * bdtri domain     k < 0, n <= k         0.0
 *                  x < 0, x > 1
 */

  public static double bdtri( int k, int n, double y )
         throws MyArithmeticException {
    double dk, dn, p;

    if( (y < 0.0) || (y > 1.0) || (k < 0) || (n <= k) )
      throw new MyArithmeticException();

    dn = n - k;
    if( k == 0 ) {
      if( y > 0.8 )
        p = -expm1( log1p(y-1.0) / dn );
      else
        p = 1.0 - Math.pow( y, 1.0/dn );
    }
    else {
      dk = k + 1;
      p = incbet( dn, dk, 0.5 );
      if( p > 0.5 )
        p = incbi( dk, dn, 1.0-y );
      else
        p = 1.0 - incbi( dn, dk, y );
    }
    return( p );
  }
/*                                                      unity.c
 *
 * Relative error approximations for function arguments near
 * unity.
 *
 *    log1p(x) = log(1+x)
 *    expm1(x) = exp(x) - 1
 *    cosm1(x) = cos(x) - 1
 *
 */

/* log1p(x) = log(1 + x)  */



  private static double log1p(double x) {
    /* Coefficients for log(1+x) = x - x**2/2 + x**3 P(x)/Q(x)
     * 1/sqrt(2) <= x < sqrt(2)
     * Theoretical peak relative error = 2.32e-20
     */
    final double LP[] = {
       4.5270000862445199635215E-5d,
       4.9854102823193375972212E-1d,
       6.5787325942061044846969E0d,
       2.9911919328553073277375E1d,
       6.0949667980987787057556E1d,
       5.7112963590585538103336E1d,
       2.0039553499201281259648E1d,
    };
    final double LQ[] = {
      /* 1.0000000000000000000000E0d,*/
       1.5062909083469192043167E1d,
       8.3047565967967209469434E1d,
       2.2176239823732856465394E2d,
       3.0909872225312059774938E2d,
       2.1642788614495947685003E2d,
       6.0118660497603843919306E1d,
    };
    double z;

    z = 1.0 + x;
    if( (z < SQRTH) || (z > SQRT2) )
      return( Math.log(z) );
    z = x*x;
    z = -0.5 * z + x * ( z * polevl( x, LP ) / p1evl( x, LQ ) );
    return (x + z);
  }

/* expm1(x) = exp(x) - 1  */


  private static double expm1(double x) {
    /*  e^x =  1 + 2x P(x^2)/( Q(x^2) - P(x^2) )
     * -0.5 <= x <= 0.5
     */

    final double EP[] = {
       1.2617719307481059087798E-4d,
       3.0299440770744196129956E-2d,
       9.9999999999999999991025E-1d,
    };
    final double EQ[] = {
       3.0019850513866445504159E-6d,
       2.5244834034968410419224E-3d,
       2.2726554820815502876593E-1d,
       2.0000000000000000000897E0d,
    };
    double r, xx;

    if( (x < -0.5) || (x > 0.5) )
      return( Math.exp(x) - 1.0 );
    xx = x * x;
    r = x * polevl( xx, EP );
    r = r/( polevl( xx, EQ ) - r );
    return (r + r);
  }

/* cosm1(x) = cos(x) - 1  */


  private static double cosm1(double x) {
    final double coscof[] = {
       4.7377507964246204691685E-14d,
      -1.1470284843425359765671E-11d,
       2.0876754287081521758361E-9d,
      -2.7557319214999787979814E-7d,
       2.4801587301570552304991E-5d,
      -1.3888888888888872993737E-3d,
       4.1666666666666666609054E-2d,
    };
    double xx;

    if( (x < -PIO4) || (x > PIO4) )
      return( Math.cos(x) - 1.0 );
    xx = x * x;
    xx = -0.5*xx + xx * xx * polevl( xx, coscof );
    return xx;
  }

/*                                                      nbdtr.c
 *
 *      Negative binomial distribution
 *
 *
 *
 * SYNOPSIS:
 *
 * int k, n;
 * double p, y, nbdtr();
 *
 * y = nbdtr( k, n, p );
 *
 * DESCRIPTION:
 *
 * Returns the sum of the terms 0 through k of the negative
 * binomial distribution:
 *
 *   k
 *   --  ( n+j-1 )   n      j
 *   >   (       )  p  (1-p)
 *   --  (   j   )
 *  j=0
 *
 * In a sequence of Bernoulli trials, this is the probability
 * that k or fewer failures precede the nth success.
 *
 * The terms are not computed individually; instead the incomplete
 * beta integral is employed, according to the formula
 *
 * y = nbdtr( k, n, p ) = incbet( n, k+1, p ).
 *
 * The arguments must be positive, with p ranging from 0 to 1.
 *
 * ACCURACY:
 *
 * Tested at random points (a,b,p), with p between 0 and 1.
 *
 *               a,b                     Relative error:
 * arithmetic  domain     # trials      peak         rms
 *    IEEE     0,100       100000      1.7e-13     8.8e-15
 * See also incbet.c.
 *
 */

  public static double nbdtr( int k, int n, double p )
         throws MyArithmeticException {

    if( (p < 0.0) || (p > 1.0) || ( k < 0 ))
      throw new MyArithmeticException();
    return( incbet( n, k+1, p ) );
  }

/*                                                      nbdtrc.c
 *
 *      Complemented negative binomial distribution
 *
 *
 *
 * SYNOPSIS:
 *
 * int k, n;
 * double p, y, nbdtrc();
 *
 * y = nbdtrc( k, n, p );
 *
 * DESCRIPTION:
 *
 * Returns the sum of the terms k+1 to infinity of the negative
 * binomial distribution:
 *
 *   inf
 *   --  ( n+j-1 )   n      j
 *   >   (       )  p  (1-p)
 *   --  (   j   )
 *  j=k+1
 *
 * The terms are not computed individually; instead the incomplete
 * beta integral is employed, according to the formula
 *
 * y = nbdtrc( k, n, p ) = incbet( k+1, n, 1-p ).
 *
 * The arguments must be positive, with p ranging from 0 to 1.
 *
 * ACCURACY:
 *
 * Tested at random points (a,b,p), with p between 0 and 1.
 *
 *               a,b                     Relative error:
 * arithmetic  domain     # trials      peak         rms
 *    IEEE     0,100       100000      1.7e-13     8.8e-15
 * See also incbet.c.
 */

  public static double nbdtrc( int k, int n, double p )
         throws MyArithmeticException {

    if( (p < 0.0) || (p > 1.0) || (k<0))
      throw new MyArithmeticException();
    return( incbet( k+1, n, 1.0 - p ) );
  }

/*                                                      nbdtri
 *
 *      Functional inverse of negative binomial distribution
 *
 *
 *
 * SYNOPSIS:
 *
 * int k, n;
 * double p, y, nbdtri();
 *
 * p = nbdtri( k, n, y );
 *
 * DESCRIPTION:
 *
 * Finds the argument p such that nbdtr(k,n,p) is equal to y.
 *
 * ACCURACY:
 *
 * Tested at random points (a,b,y), with y between 0 and 1.
 *
 *               a,b                     Relative error:
 * arithmetic  domain     # trials      peak         rms
 *    IEEE     0,100       100000      1.5e-14     8.5e-16
 * See also incbi.c.
 */

  public static double nbdtri( int k, int n, double p )
         throws MyArithmeticException {
    if( (p < 0.0) || (p > 1.0) || (k<0))
      throw new MyArithmeticException();
    return incbi( n, k+1, p );
  }


}


