// Author: Manfred Georg

import java.awt.Color;

// This class defines a set of data.  The data is either an array of
// DoublePoint or an array of coefficients for a polynomial
// 

public class Plot extends Object {
  protected int type; // 0 = points 1 = graph, 2 = repeating data points
  protected DoublePoint data[];
  protected double coef[], period;
  protected Color color;

  public Plot(DoublePoint dataPoints[], Color c) {
    type = 0;
    data = new DoublePoint[dataPoints.length];
    for(int x=0; x<data.length; x++) {
      data[x] = (DoublePoint)dataPoints[x].clone();
    }
    color = c;
  }

  public Plot(DoublePoint dataPoints[], Color c, double period) {
    type = 2;
    data = new DoublePoint[dataPoints.length];
    for(int x=0; x<data.length; x++) {
      data[x] = (DoublePoint)dataPoints[x].clone();
    }
    this.period = period;
    color = c;
  }

  private Plot(DoublePoint dataPoints[], Color c, boolean extra) {
    // more effecent constructor
    type = 0;
    data = dataPoints;
    color = c;
  }

  private Plot(DoublePoint dataPoints[], Color c,
               double period, boolean extra) {
    // more effecent constructor
    type = 2;
    data = dataPoints;
    this.period = period;
    color = c;
  }

  public Plot(double c[], Color color) {
    type = 1;
    // coef[0]+x*(coef[1]+x*(...))
    coef = new double[c.length];
    for(int x=0; x<c.length; x++) {
      coef[x] = c[x];
    }
    this.color = color;
  }

  public Plot(double c[], Color color, boolean extra) {
    // more effeciant
    type = 1;
    // coef[0]+x*(coef[1]+x*(...))
    coef = c;
    this.color = color;
  }

  public int getType() {
    return type;
  }
  
  public DoublePoint getDataPoint(int point) {
    return data[point];
  }
  
  public MaxMin getMaxMin() {
    if(type==1) // polynomial
      return null;
    // periodic or none periodic
    // if periodic get length of one period
    // could be shortcutted to look at first and last data points for x
    // value...but I won't in case someone decides to be crazy.
    // and anyway we need to look at them all for y values now

    MaxMin mm = new MaxMin();

    mm.minX = mm.maxX = data[0].x;
    mm.minY = mm.maxY = data[0].y;

    for(int i=1;i<data.length;i++) {
      if(mm.minX>data[i].x)
        mm.minX = data[i].x;
      else if(mm.maxX<data[i].x)
        mm.maxX = data[i].x;
      if(mm.minY>data[i].y)
        mm.minY = data[i].y;
      else if(mm.maxY<data[i].y)
        mm.maxY = data[i].y;
    }
    
    return mm;
  }

  public double getValue(double x) {
    // coef[0]+x*(coef[1]+x*(...))
    double value=coef[coef.length-1];
    for(int i=coef.length-2; i>=0; i--) {
      value = coef[i]+x*value;
    }
    return value;
  }
  
  public int size() {
    return data.length;
  }

  public Color getColor() {
    return color;
  }

  public void setColor(Color c) {
    color = c;
  }

  public double getPeriod() {
    return period;
  }

  // Method by Andy Leach

  public double[] getCoef() {
    return coef;
  }

  // This method creates another Plot which is the seasonal component
  // of the instance

  public Plot getSeason(double p, Color color) {
    double season[];
    DoublePoint out[];
    int n, halfN; // number of data points in a period
    double temp;
    if(p <= 1e-12) { // == 0
      return new Plot(this.data,color,true); // returns itself (new color)
    }
    for(n=1; data[n].x-data[0].x != p; n++); // find the number of data
       // points in a period
    if(n % 2 == 0) { // even number of data points in a period
      halfN = n/2;
      season = new double[data.length-n];
      for(int i=halfN; i+halfN < data.length; i++) {
        season[i-halfN]=data[i].y;
        int j;
        for(j=1; j < halfN; j++) {
          season[i-halfN]+=data[i-j].y+data[i+j].y;
        }
        season[i-halfN] += (data[i-j].y+data[i+j].y)/2d;
        season[i-halfN] /= (double)n;
        season[i-halfN] = data[i].y - season[i-halfN];
      }
      out = new DoublePoint[n];
      for(int i=0, j; i < halfN; i++) { // average the periods
        out[i] = new DoublePoint(data[i].x,0);
        for(j = 0; i+halfN+j*n < season.length; j++) {
          out[i].y += season[i+halfN+j*n];
        }
        out[i].y /= ((double)j);
      }
      for(int i=halfN, j; i < n; i++) { // average the periods
        out[i] = new DoublePoint(data[i].x,0);
        for(j = 0; i-halfN+j*n < season.length; j++) {
          out[i].y += season[i-halfN+j*n];
        }
        out[i].y /= ((double)j);
      }
      temp=0;
      for(int i=0; i < n; i++ ) {
        temp += out[i].y;
      }
      temp /= (double)n;
      for(int i=0; i < n; i++ ) {
        out[i].y -= temp;
      }
      return new Plot(out, color, p, true);
    } else {  // odd number of points in a period
      halfN = (n-1)/2;
      season = new double[data.length-n+1];
      //out = new DoublePoint[season.length];
      for(int i=halfN; i+halfN < data.length; i++) {
        season[i-halfN]=data[i].y;
        int j;
        for(j=1; j <= halfN; j++) {
          season[i-halfN]+=data[i-j].y+data[i+j].y;
        }
        season[i-halfN] /= (double)n;
        season[i-halfN] = data[i].y - season[i-halfN];
        //System.out.println("season["+(i-halfN)+"] = "+season[i-halfN]);
        //out[i-halfN] = new DoublePoint(data[i].x,season[i-halfN]);
      }
      //return new Plot(out, color, true);
      out = new DoublePoint[n];
      for(int i=0, j; i < halfN; i++) { // average the periods
        out[i] = new DoublePoint(data[i].x,0);
        //System.out.println("first: i = "+(i+halfN+1));
        for(j = 0; i+halfN+1+j*n < season.length; j++) {
          out[i].y += season[i+halfN+1+j*n];
        }
        out[i].y /= ((double)j);
      }
      for(int i=halfN, j; i < n; i++) { // average the periods
        out[i] = new DoublePoint(data[i].x,0);
        //System.out.println("second: i = "+(i-halfN));
        for(j = 0; i-halfN+j*n < season.length; j++) {
          out[i].y += season[i-halfN+j*n];
        }
        out[i].y /= ((double)j);
      }
      temp=0;
      for(int i=0; i < n; i++ ) {
        temp += out[i].y;
      }
      temp /= (double)n;
      for(int i=0; i < n; i++ ) {
        out[i].y -= temp;
      }
      return new Plot(out, color, p, true);
    }
  }

  // This method takes a Plot (season) and subtracts it from the instance
  // and returns the output as a Plot

  public Plot deSeasonalize(Plot season, Color c) {
    // assume they are on the same x step size
    DoublePoint newData[] = new DoublePoint[data.length];
    for(int x=0, i; x < newData.length;) {
      for(i=0;i<season.size() && x<newData.length;i++,x++) {
        newData[x] =
           new DoublePoint(data[x].x,data[x].y-season.getDataPoint(i).y);
      }
    }
    return new Plot(newData,c,true);
  }

  // This method takes a Plot (poly) which must be a polynomial plot
  // and subtracts if from the instance and returns the output as a Plot

  public Plot subtract(Plot poly, Color c) {
    DoublePoint newData[] = new DoublePoint[data.length];
    for(int x=0; x < newData.length;x++) {
      newData[x] =
         new DoublePoint(data[x].x,data[x].y-poly.getValue(data[x].x));
    }
    return new Plot(newData,c,true);
  }

  /*
   * converted from Matlab by Manfred Georg
   * original program by Kurt Georg
% polynomial least squares fit
%
% input:
% x , y are the data vectors
% n is the length of the data
%
% d is the degree of the polynomial
% p(x) = c(1)*x^0 + ... + c(d+1)*x^d
% to fit the data
%
% output: the coefficients c of the polynomial,
% a vector of length d+1
%
% Note that MATLAB arrays and columns begin their numbering
% with 1, whereas C begins usually with 0 !!!!!!!!!!!!!
   * java arrays also begin with 0 of course
%
   *
   */
  

  public Plot leastSquaresFit(int d, Color color) {
    // % Setting up the vectors and arrays:
    double c[] = new double[d+1]; // coef of new Plot
    double A[][] = new double[d+1][];
    for(int i=0; i<d+1; i++)
      A[i] = new double[d+1];
    double b[] = new double[d+1];
    double s[] = new double[2*d+1];
    // % loading the normal equations A*c=b:
    for(int i=0,k; i<=(2*d) ; i++) {
      s[i] = 0;
      for(k=0; k<data.length ; k++) {
        s[i] += myPow(data[k].x,i);
      }
    }
    for(int p=0; p <= d; p++) {
      for(int q=p; q<=d; q++) {
        A[p][q] = s[p+q];
      }
    }
    for(int i=0; i<=d; i++) {
      b[i]=0;
      for(int k=0; k<data.length; k++) {
        b[i] += myPow(data[k].x,i)*data[k].y; // could have saved the x^i
      }
    }
    // % solving the normal equations via Cholesky for c
    for(int i=0,k; i<=d; i++) {
      for(k=0;k<i;k++) {
        A[i][i] -= A[k][i]*A[k][i];
      }
      A[i][i] = Math.sqrt(A[i][i]);
      for(int l=i+1; l<=d; l++) {
        for(k=0;k<i;k++) {
	  A[i][l] -= A[k][i]*A[k][l];
	}
	A[i][l] = A[i][l]/A[i][i];
      }
    }
    // % forward solving
    for(int k=0,i;k<=d;k++) {
      c[k]=b[k];
      for(i=0;i<k;i++) {
        c[k] -= A[i][k]*c[i];
      }
      c[k] /= A[k][k];
    }
    // % back solving
    for(int k=d,i; k>=0; k--) {
      for(i=k+1; i<=d; i++) {
        c[k] -= A[k][i]*c[i];
      }
      c[k] /= A[k][k];
    }
    return new Plot(c, color,true);
  }

  private double myPow(double x, int i) { // calculate x^i
    double value = 1;
    for(int j=1; j<=i; j++) {
      value *= x;
    }
    return value;
  }

  public String toString() {
    StringBuffer sb = new StringBuffer();
    switch(type) {
    case 2:
    case 0:
      sb.append("Plot[type="+type+",\n");
      for(int i=0; i < data.length; i++) {
        sb.append(i).append(":\t").append(data[i].toString()).append(",\n");
      }
      sb.append(color).append("]");
      return sb.toString();
    case 1:
      sb.append("Plot[type="+type+",");
      for(int i=0; i < coef.length; i++) {
        sb.append("coef[").append(i).append("]=").append(coef[i]).append(",");
      }
      sb.append(color).append("]");
      return sb.toString();
    default:
      return "Plot[OOPS]";
    }
  }
}
